<?php
/**
 * Trackveil Dashboard Widget
 * Shows analytics stats in WordPress admin dashboard
 */

if (!defined('ABSPATH')) {
    exit;
}

class Trackveil_Dashboard_Widget {
    
    public static function init() {
        add_action('wp_dashboard_setup', [__CLASS__, 'register_widget']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_styles']);
    }
    
    /**
     * Register dashboard widget
     */
    public static function register_widget() {
        // Only show if site ID is configured
        if (!get_option('trackveil_site_id')) {
            return;
        }
        
        wp_add_dashboard_widget(
            'trackveil_stats',
            __('📊 Trackveil Analytics', 'trackveil-analytics'),
            [__CLASS__, 'render_widget']
        );
    }
    
    /**
     * Enqueue widget styles
     */
    public static function enqueue_styles($hook) {
        if ('index.php' !== $hook) {
            return;
        }
        
        // Inline minimal CSS
        wp_add_inline_style('dashboard', '
            #trackveil_stats .trackveil-stat { display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #ddd; }
            #trackveil_stats .trackveil-stat:last-child { border-bottom: none; }
            #trackveil_stats .trackveil-value { font-weight: 600; color: #2DD4BF; }
            #trackveil_stats .trackveil-page { font-size: 13px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
            #trackveil_stats .trackveil-current { color: #10b981; font-weight: 600; }
            #trackveil_stats .trackveil-error { color: #dc2626; padding: 12px; background: #fee; border-radius: 4px; }
        ');
    }
    
    /**
     * Render widget content
     */
    public static function render_widget() {
        $site_id = get_option('trackveil_site_id');
        $api_key = get_option('trackveil_api_key'); // Optional - for Premium+ users
        
        // If no API key, show upgrade message
        if (empty($api_key)) {
            ?>
            <div style="padding: 12px; background: #f0f9ff; border: 1px solid #0ea5e9; border-radius: 4px;">
                <p style="margin: 0 0 8px 0; font-weight: 600;"><?php _e('✨ Want to see stats here?', 'trackveil-analytics'); ?></p>
                <p style="margin: 0 0 8px 0; font-size: 13px;">
                    <?php _e('Generate an API key in your Trackveil dashboard to display analytics right here in WordPress.', 'trackveil-analytics'); ?>
                </p>
                <a href="<?php echo esc_url(TRACKVEIL_DASHBOARD . '/app/api-keys'); ?>" target="_blank" class="button button-primary button-small">
                    <?php _e('Get API Key', 'trackveil-analytics'); ?>
                </a>
            </div>
            <?php
            return;
        }
        
        // Fetch stats from API (cached for 5 minutes)
        $stats = self::fetch_stats($site_id, $api_key);
        
        if (is_wp_error($stats)) {
            ?>
            <div class="trackveil-error">
                <strong><?php _e('⚠️ Error loading stats:', 'trackveil-analytics'); ?></strong> <?php echo esc_html($stats->get_error_message()); ?>
            </div>
            <?php
            return;
        }
        
        ?>
        <div style="padding: 4px 0;">
            <!-- Today's Stats -->
            <div class="trackveil-stat">
                <span><?php _e("Today's Views:", 'trackveil-analytics'); ?></span>
                <span class="trackveil-value"><?php echo number_format($stats['page_views']); ?></span>
            </div>
            <div class="trackveil-stat">
                <span><?php _e('Unique Visitors:', 'trackveil-analytics'); ?></span>
                <span class="trackveil-value"><?php echo number_format($stats['unique_visitors']); ?></span>
            </div>
            <div class="trackveil-stat">
                <span><?php _e('Current Visitors:', 'trackveil-analytics'); ?></span>
                <span class="trackveil-current">🟢 <?php echo esc_html($stats['current_visitors']); ?></span>
            </div>
            
            <?php if (!empty($stats['top_pages'])): ?>
                <div style="margin-top: 12px; padding-top: 12px; border-top: 2px solid #eee;">
                    <strong style="font-size: 13px;"><?php _e('Top Pages (Last 7 Days)', 'trackveil-analytics'); ?></strong>
                    <div style="margin-top: 8px;">
                        <?php foreach (array_slice($stats['top_pages'], 0, 5) as $page): ?>
                            <div class="trackveil-page" style="padding: 4px 0;">
                                <span style="color: #666;"><?php echo esc_html($page['page_url']); ?></span>
                                <span style="float: right; color: #2DD4BF; font-weight: 600;">
                                    <?php echo number_format($page['views']); ?>
                                </span>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>
            
            <div style="margin-top: 16px; padding-top: 12px; border-top: 1px solid #eee; text-align: center;">
                <a href="<?php echo esc_url(TRACKVEIL_DASHBOARD); ?>" target="_blank" class="button button-small">
                    <?php _e('View Full Dashboard →', 'trackveil-analytics'); ?>
                </a>
            </div>
        </div>
        <?php
    }
    
    /**
     * Fetch stats from Trackveil API
     * Cached for 5 minutes to reduce API calls
     */
    private static function fetch_stats($site_id, $api_key) {
        // Check cache first
        $cache_key = 'trackveil_stats_' . $site_id;
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        // Fetch today's stats
        $stats_response = wp_remote_get(
            TRACKVEIL_API . '/v1/sites/' . $site_id . '/stats?start_date=' . date('Y-m-d') . '&end_date=' . date('Y-m-d'),
            [
                'headers' => ['Authorization' => 'Bearer ' . $api_key],
                'timeout' => 10
            ]
        );
        
        if (is_wp_error($stats_response)) {
            return $stats_response;
        }
        
        $stats_data = json_decode(wp_remote_retrieve_body($stats_response), true);
        
        // Handle new nested response format
        $stats = $stats_data['stats'] ?? $stats_data; // Backwards compatible
        $limits = $stats_data['limits'] ?? [];
        
        // Fetch top pages (last 7 days)
        $pages_response = wp_remote_get(
            TRACKVEIL_API . '/v1/sites/' . $site_id . '/pages?start_date=' . date('Y-m-d', strtotime('-7 days')) . '&limit=5',
            [
                'headers' => ['Authorization' => 'Bearer ' . $api_key],
                'timeout' => 10
            ]
        );
        
        if (!is_wp_error($pages_response)) {
            $pages_data = json_decode(wp_remote_retrieve_body($pages_response), true);
            $stats['top_pages'] = $pages_data['pages'] ?? [];
        }
        
        // Cache for 5 minutes
        set_transient($cache_key, $stats, 5 * MINUTE_IN_SECONDS);
        
        return $stats;
    }
}

