<?php
/**
 * Trackveil API Class
 * Handles communication with Trackveil API
 */

if (!defined('ABSPATH')) {
    exit;
}

class Trackveil_API {
    
    public static function init() {
        // API methods are called directly, no hooks needed
    }
    
    /**
     * Create Trackveil account from WordPress
     * Uses HMAC signature for security
     */
    public static function create_account($email, $name) {
        $site_url = home_url();
        $site_name = get_bloginfo('name');
        
        // Prepare request data
        $data = [
            'email' => $email,
            'name' => $name,
            'site_name' => $site_name,
            'site_url' => $site_url,
            'wp_version' => get_bloginfo('version'),
            'wp_home_url' => home_url(),
            'wp_admin_email' => get_option('admin_email'),
            'plugin_version' => TRACKVEIL_VERSION,
            'timestamp' => time()
        ];
        
        $json = json_encode($data);
        
        // Make API request
        // Note: API validates WordPress requests using timestamp, domain validation,
        // and rate limiting instead of HMAC signature
        $response = wp_remote_post(TRACKVEIL_API . '/v1/wordpress/signup', [
            'headers' => [
                'Content-Type' => 'application/json',
                'X-WordPress-Plugin' => 'trackveil-analytics/' . TRACKVEIL_VERSION
            ],
            'body' => $json,
            'timeout' => 30
        ]);
        
        if (is_wp_error($response)) {
            return [
                'success' => false,
                /* translators: %s: Error message from the server */
                'error' => sprintf(__('Connection failed: %s', 'trackveil-analytics'), $response->get_error_message())
            ];
        }
        
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 201) {
            return [
                'success' => false,
                'error' => $result['message'] ?? __('Signup failed', 'trackveil-analytics')
            ];
        }
        
        return $result;
    }
    
    /**
     * Verify site ID is valid
     */
    public static function verify_site_id($site_id) {
        if (empty($site_id) || strlen($site_id) !== 32) {
            return false;
        }
        
        // Site ID should be alphanumeric
        return ctype_alnum($site_id);
    }
}

