<?php
/**
 * Trackveil Admin Class
 * Handles WordPress admin interface and settings
 */

if (!defined('ABSPATH')) {
    exit;
}

class Trackveil_Admin {
    
    public static function init() {
        add_action('admin_menu', [__CLASS__, 'add_menu']);
        add_action('admin_init', [__CLASS__, 'register_settings']);
        add_action('admin_post_trackveil_create_account', [__CLASS__, 'handle_create_account']);
    }
    
    /**
     * Add settings page to WordPress admin
     */
    public static function add_menu() {
        add_options_page(
            __('Trackveil Analytics', 'trackveil-analytics'),
            __('Trackveil', 'trackveil-analytics'),
            'manage_options',
            'trackveil',
            [__CLASS__, 'render_settings_page']
        );
    }
    
    /**
     * Register plugin settings
     */
    public static function register_settings() {
        register_setting('trackveil', 'trackveil_site_id', [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ]);
        
        register_setting('trackveil', 'trackveil_skip_admins', [
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default' => true
        ]);
        
        register_setting('trackveil', 'trackveil_api_key', [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ]);
    }
    
    /**
     * Handle account creation from plugin
     */
    public static function handle_create_account() {
        check_admin_referer('trackveil_create_account');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Unauthorized', 'trackveil-analytics'));
        }
        
        $email = sanitize_email($_POST['email']);
        $name = sanitize_text_field($_POST['name']);
        
        if (empty($email) || empty($name)) {
            wp_redirect(add_query_arg(['page' => 'trackveil', 'error' => 'missing_fields'], admin_url('options-general.php')));
            exit;
        }
        
        // Call API to create account
        $result = Trackveil_API::create_account($email, $name);
        
        if (!$result['success']) {
            wp_redirect(add_query_arg([
                'page' => 'trackveil',
                'error' => urlencode($result['error'])
            ], admin_url('options-general.php')));
            exit;
        }
        
        // Save site ID and account email
        update_option('trackveil_site_id', $result['site_id']);
        update_option('trackveil_account_email', $email);
        
        // Store temp password in transient (expires in 1 hour for security)
        set_transient('trackveil_temp_password', $result['temp_password'], HOUR_IN_SECONDS);
        
        // Auto-save API key if provided
        if (!empty($result['api_key'])) {
            update_option('trackveil_api_key', $result['api_key']);
        }
        
        wp_redirect(add_query_arg([
            'page' => 'trackveil',
            'success' => '1'
        ], admin_url('options-general.php')));
        exit;
    }
    
    /**
     * Render settings page
     */
    public static function render_settings_page() {
        $site_id = get_option('trackveil_site_id');
        $temp_password = get_transient('trackveil_temp_password'); // Expires after 1 hour
        $account_email = get_option('trackveil_account_email');
        $skip_admins = get_option('trackveil_skip_admins', true);
        
        $error = isset($_GET['error']) ? urldecode($_GET['error']) : '';
        $success = isset($_GET['success']);
        
        ?>
        <div class="wrap">
            <h1><?php _e('Trackveil Analytics', 'trackveil-analytics'); ?></h1>
            
            <?php if ($error): ?>
                <div class="notice notice-error">
                    <p><?php echo esc_html($error); ?></p>
                </div>
            <?php endif; ?>
            
            <?php if ($success && $temp_password): ?>
                <div class="notice notice-success">
                    <p><strong><?php _e('Account created successfully!', 'trackveil-analytics'); ?></strong></p>
                    <p>
                        <?php _e('Your temporary password:', 'trackveil-analytics'); ?> 
                        <code style="background: #f0f0f0; padding: 4px 8px; border-radius: 3px; font-size: 14px;"><?php echo esc_html($temp_password); ?></code>
                    </p>
                    <p>
                        <?php _e('Save this password and login at', 'trackveil-analytics'); ?> 
                        <a href="<?php echo esc_url(TRACKVEIL_DASHBOARD); ?>" target="_blank"><?php echo esc_url(TRACKVEIL_DASHBOARD); ?></a>
                    </p>
                    <p><?php _e('Check your email for account details. Tracking is now active!', 'trackveil-analytics'); ?></p>
                </div>
            <?php endif; ?>
            
            <?php if (empty($site_id)): ?>
                <!-- Setup Screen -->
                <div class="card" style="max-width: 600px;">
                    <h2><?php _e('Connect to Trackveil', 'trackveil-analytics'); ?></h2>
                    
                    <!-- Option A: Create Account -->
                    <div style="border: 2px solid #2DD4BF; border-radius: 8px; padding: 20px; margin-bottom: 20px; background: #f0fdfa;">
                        <h3 style="margin-top: 0; color: #0d9488;">✨ <?php _e('New to Trackveil? Create Account', 'trackveil-analytics'); ?></h3>
                        <p><?php _e('Create your free Trackveil account right here - no need to leave WordPress!', 'trackveil-analytics'); ?></p>
                        
                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                            <?php wp_nonce_field('trackveil_create_account'); ?>
                            <input type="hidden" name="action" value="trackveil_create_account">
                            
                            <table class="form-table" role="presentation">
                                <tr>
                                    <th scope="row">
                                        <label for="trackveil_email"><?php _e('Your Email', 'trackveil-analytics'); ?></label>
                                    </th>
                                    <td>
                                        <input type="email" 
                                               id="trackveil_email" 
                                               name="email" 
                                               class="regular-text" 
                                               required
                                               value="<?php echo esc_attr(wp_get_current_user()->user_email); ?>"
                                               placeholder="you@example.com">
                                        <p class="description"><?php _e('We\'ll create your Trackveil account with this email', 'trackveil-analytics'); ?></p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row">
                                        <label for="trackveil_name"><?php _e('Your Name', 'trackveil-analytics'); ?></label>
                                    </th>
                                    <td>
                                        <input type="text" 
                                               id="trackveil_name" 
                                               name="name" 
                                               class="regular-text" 
                                               required
                                               value="<?php echo esc_attr(wp_get_current_user()->display_name); ?>"
                                               placeholder="John Doe">
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php _e('Site Details', 'trackveil-analytics'); ?></th>
                                    <td>
                                        <p><strong><?php _e('Site Name:', 'trackveil-analytics'); ?></strong> <?php echo esc_html(get_bloginfo('name')); ?></p>
                                        <p><strong><?php _e('Site URL:', 'trackveil-analytics'); ?></strong> <?php echo esc_url(home_url()); ?></p>
                                        <p class="description"><?php _e('These will be automatically added to your Trackveil account', 'trackveil-analytics'); ?></p>
                                    </td>
                                </tr>
                            </table>
                            
                            <p class="submit">
                                <button type="submit" class="button button-primary button-hero">
                                    <?php _e('Create Free Account & Start Tracking', 'trackveil-analytics'); ?>
                                </button>
                            </p>
                            
                            <p style="font-size: 12px; color: #666;">
                                <?php _e('By creating an account, you agree to Trackveil\'s', 'trackveil-analytics'); ?> 
                                <a href="https://trackveil.net/terms" target="_blank"><?php _e('Terms of Service', 'trackveil-analytics'); ?></a> 
                                <?php _e('and', 'trackveil-analytics'); ?> 
                                <a href="https://trackveil.net/privacy" target="_blank"><?php _e('Privacy Policy', 'trackveil-analytics'); ?></a>
                            </p>
                        </form>
                    </div>
                    
                    <p style="text-align: center; margin: 20px 0; color: #666;">— <?php _e('OR', 'trackveil-analytics'); ?> —</p>
                    
                    <!-- Option B: Enter Site ID -->
                    <div style="border: 1px solid #ddd; border-radius: 8px; padding: 20px;">
                        <h3 style="margin-top: 0;"><?php _e('Already have a Trackveil account?', 'trackveil-analytics'); ?></h3>
                        <p><?php _e('Enter your Site ID from your Trackveil dashboard', 'trackveil-analytics'); ?></p>
                        
                        <form method="post" action="options.php">
                            <?php settings_fields('trackveil'); ?>
                            
                            <table class="form-table" role="presentation">
                                <tr>
                                    <th scope="row">
                                        <label for="trackveil_site_id"><?php _e('Site ID', 'trackveil-analytics'); ?></label>
                                    </th>
                                    <td>
                                        <input type="text" 
                                               id="trackveil_site_id" 
                                               name="trackveil_site_id" 
                                               class="regular-text" 
                                               value="<?php echo esc_attr($site_id); ?>"
                                               placeholder="32-character site ID"
                                               pattern="[a-zA-Z0-9]{32}">
                                        <p class="description">
                                            <?php _e('Find this at', 'trackveil-analytics'); ?> 
                                            <a href="<?php echo esc_url(TRACKVEIL_DASHBOARD . '/app/sites'); ?>" target="_blank">
                                                <?php _e('dashboard.trackveil.net/app/sites', 'trackveil-analytics'); ?>
                                            </a>
                                        </p>
                                    </td>
                                </tr>
                            </table>
                            
                            <?php submit_button(__('Save Site ID', 'trackveil-analytics')); ?>
                        </form>
                    </div>
                </div>
                
            <?php else: ?>
                <!-- Already Configured -->
                <div class="notice notice-success inline" style="margin: 20px 0; padding: 12px;">
                    <p><strong><?php _e('✅ Trackveil is active and tracking your site!', 'trackveil-analytics'); ?></strong></p>
                </div>
                
                <div class="card" style="max-width: 600px;">
                    <h2><?php _e('Settings', 'trackveil-analytics'); ?></h2>
                    
                    <form method="post" action="options.php">
                        <?php settings_fields('trackveil'); ?>
                        
                        <table class="form-table" role="presentation">
                            <tr>
                                <th scope="row">
                                    <label for="trackveil_site_id"><?php _e('Site ID', 'trackveil-analytics'); ?></label>
                                </th>
                                <td>
                                    <input type="text" 
                                           id="trackveil_site_id" 
                                           name="trackveil_site_id" 
                                           class="regular-text" 
                                           value="<?php echo esc_attr($site_id); ?>"
                                           readonly
                                           style="background: #f0f0f0;">
                                    <p class="description">
                                        <?php _e('View your analytics at', 'trackveil-analytics'); ?> 
                                        <a href="<?php echo esc_url(TRACKVEIL_DASHBOARD); ?>" target="_blank">
                                            <?php _e('dashboard.trackveil.net', 'trackveil-analytics'); ?>
                                        </a>
                                    </p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <?php _e('Exclude Admin Users', 'trackveil-analytics'); ?>
                                </th>
                                <td>
                                    <label>
                                        <input type="checkbox" 
                                               name="trackveil_skip_admins" 
                                               value="1" 
                                               <?php checked($skip_admins, true); ?>>
                                        <?php _e('Don\'t track logged-in administrators', 'trackveil-analytics'); ?>
                                    </label>
                                    <p class="description"><?php _e('Recommended to avoid tracking your own visits', 'trackveil-analytics'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="trackveil_api_key"><?php _e('API Key (Optional)', 'trackveil-analytics'); ?></label>
                                </th>
                                <td>
                                    <input type="text" 
                                           id="trackveil_api_key" 
                                           name="trackveil_api_key" 
                                           class="regular-text" 
                                           value="<?php echo esc_attr(get_option('trackveil_api_key')); ?>"
                                           placeholder="tv_...">
                                    <p class="description">
                                        <?php _e('Add an API key to show analytics in your WordPress dashboard widget.', 'trackveil-analytics'); ?>
                                        <a href="<?php echo esc_url(TRACKVEIL_DASHBOARD . '/app/api-keys'); ?>" target="_blank">
                                            <?php _e('Generate API Key →', 'trackveil-analytics'); ?>
                                        </a>
                                    </p>
                                </td>
                            </tr>
                        </table>
                        
                        <?php submit_button(); ?>
                    </form>
                    
                    <hr>
                    
                    <h3><?php _e('Installation Code', 'trackveil-analytics'); ?></h3>
                    <p><?php _e('The tracking code is automatically added to your site. No manual installation needed!', 'trackveil-analytics'); ?></p>
                    <pre style="background: #282c34; color: #abb2bf; padding: 15px; border-radius: 5px; overflow-x: auto;"><code>&lt;script async src="<?php echo esc_url(TRACKVEIL_CDN); ?>" 
        data-site-id="<?php echo esc_attr($site_id); ?>"&gt;&lt;/script&gt;</code></pre>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
}

